<?php
//require_once(dirname(__FILE__) . '/config.php');

class DashboardCache {
    private $cachePath;
    private $cacheFile;
    private $cacheLifetime = 3600; // 5 minutes in seconds

    public function __construct() {
        $this->cachePath = dirname(__FILE__) . '/../cache/';
        $this->cacheFile = $this->cachePath . 'dashboard_stats.json';
        
        // Create cache directory if it doesn't exist
        if (!file_exists($this->cachePath)) {
            mkdir($this->cachePath, 0755, true);
        }
    }

    public function getCachedData() {
        if ($this->isCacheValid()) {
            return json_decode(file_get_contents($this->cacheFile), true);
        }
        return null;
    }

    public function updateCache($db1, $debug = false) {
        try {
            // Get top clients data
            $query = "SELECT 
                c.Id,
                c.CompanyName,
                c.City,
                c.Country,
                COUNT(p.passport_id) as total_passports,
                (SELECT stop_at FROM tbl_clients_applications 
                    WHERE app_id = 4 AND client_id = c.Id LIMIT 1) AS package,
                (SELECT stop_at FROM tbl_clients_applications 
                    WHERE app_id = 4 AND client_id = c.Id LIMIT 1) - 
                    COUNT(p.passport_id) AS credit
            FROM tbl_clients c
            LEFT JOIN passports_1446_2024 p ON c.Id = p.client_id
            WHERE c.license_enabled = 1
            GROUP BY c.Id
            ORDER BY total_passports DESC";

            if ($debug) {
                echo $query;
            }
            
            $stmt = $db1->query($query);
            $topClients = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Get summary statistics
            $total_clients = $db1->query("SELECT COUNT(*) as total FROM tbl_clients WHERE license_enabled = 1")
                ->fetch(PDO::FETCH_ASSOC)['total'];
                
            $total_passports = $db1->query("SELECT COUNT(*) as total FROM passports_1446_2024")
                ->fetch(PDO::FETCH_ASSOC)['total'];
                
            $low_credit = $db1->query("SELECT COUNT(*) as total 
                FROM (
                    SELECT c.Id,
                        (SELECT stop_at FROM tbl_clients_applications 
                            WHERE app_id = 4 AND client_id = c.Id LIMIT 1) - 
                            COUNT(p.passport_id) as remaining_credit
                    FROM tbl_clients c
                    LEFT JOIN passports_1446_2024 p ON c.Id = p.client_id
                    WHERE c.license_enabled = 1
                    GROUP BY c.Id
                    HAVING remaining_credit < " . (defined('CREDIT_WARNING') ? CREDIT_WARNING : 50) . "
                ) as low_credit_clients")
                ->fetch(PDO::FETCH_ASSOC)['total'];

            // Get low credit clients list
            $low_credit_query = "
                SELECT 
                    c.Id,
                    c.CompanyName,
                    co.CNTRY_NAME_LA as CountryName,
                    COUNT(p.passport_id) as used_passports,
                    (SELECT stop_at FROM tbl_clients_applications 
                        WHERE app_id = 4 AND client_id = c.Id LIMIT 1) as package_limit,
                    (SELECT stop_at FROM tbl_clients_applications 
                        WHERE app_id = 4 AND client_id = c.Id LIMIT 1) - 
                        COUNT(p.passport_id) as remaining_credit
                FROM tbl_clients c
                LEFT JOIN passports_1446_2024 p ON c.Id = p.client_id
                LEFT JOIN tbl_countries co ON c.Country = co.CNTRY_ID
                WHERE c.status = 1 AND c.license_enabled = 1
                GROUP BY c.Id
                HAVING remaining_credit < " . (defined('CREDIT_WARNING') ? CREDIT_WARNING : 50) . "
                ORDER BY remaining_credit ASC";

            $low_credit_clients = $db1->query($low_credit_query)->fetchAll(PDO::FETCH_ASSOC);

            // Prepare cache data
            $cacheData = [
                'timestamp' => time(),
                'top_clients' => $topClients,
                'summary' => [
                    'total_clients' => $total_clients,
                    'total_passports' => $total_passports,
                    'low_credit' => $low_credit
                ],
                'low_credit_clients' => $low_credit_clients
            ];

            // Save to cache file
            file_put_contents($this->cacheFile, json_encode($cacheData));
            return $cacheData;

        } catch (PDOException $e) {
            error_log("Cache Update Error: " . $e->getMessage());
            return null;
        }
    }

    private function isCacheValid() {
        if (!file_exists($this->cacheFile)) {
            return false;
        }

        $cacheData = json_decode(file_get_contents($this->cacheFile), true);
        if (!isset($cacheData['timestamp'])) {
            return false;
        }

        return (time() - $cacheData['timestamp']) < $this->cacheLifetime;
    }
} 