<?php
// Include necessary configuration and database connection files
include_once("../includes/config.php");
include_once("../functions.php");

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set headers for JSON response
header('Content-Type: application/json');

// Log incoming request data
error_log("Received POST data: " . print_r($_POST, true));

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method'
    ]);
    exit;
}

// Get and validate input parameters
$app = isset($_POST['app']) ? trim($_POST['app']) : '';
$client_id = isset($_POST['client_id']) ? intval($_POST['client_id']) : 0;
$limit = isset($_POST['limit']) ? intval($_POST['limit']) : null;
$availability = isset($_POST['availability']) ? intval($_POST['availability']) : null;
$blocked = isset($_POST['blocked']) ? intval($_POST['blocked']) : null;

// Log processed parameters
error_log("Processed parameters - app: $app, client_id: $client_id, limit: $limit, availability: $availability, blocked: $blocked");

// Validate inputs
if (empty($app) || $client_id <= 0) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid parameters provided'
    ]);
    exit;
}

// Validate limit value if provided
if ($limit !== null && ($limit < 0 || $limit > 999999)) {
    echo json_encode([
        'success' => false,
        'message' => 'Package limit must be between 0 and 999999'
    ]);
    exit;
}

// Validate availability value if provided
if ($availability !== null && !in_array($availability, [0, 1])) {
    echo json_encode([
        'success' => false,
        'message' => 'Availability must be 0 or 1'
    ]);
    exit;
}

// Validate blocked value if provided
if ($blocked !== null && !in_array($blocked, [0, 1])) {
    echo json_encode([
        'success' => false,
        'message' => 'Blocked must be 0 or 1'
    ]);
    exit;
}

try {
    // Verify database connection
    if (!isset($db1)) {
        throw new Exception('Database connection not available');
    }

    // First get the App_Id from tbl_applications
    $query = "SELECT id FROM tbl_applications WHERE Application_Name = :app";
    $stmt = $db1->prepare($query);
    $stmt->bindParam(':app', $app, PDO::PARAM_STR);
    $stmt->execute();
    
    $appData = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$appData) {
        throw new Exception('Application not found');
    }
    
    $app_id = $appData['id'];

    // Build the update query based on which parameters are provided
    $updateFields = [];
    $params = [];

    if ($limit !== null) {
        $updateFields[] = "Stop_At = :limit";
        $params[':limit'] = $limit;
    }

    if ($availability !== null) {
        $updateFields[] = "availability = :availability";
        $params[':availability'] = $availability;
    }

    if ($blocked !== null) {
        $updateFields[] = "Blocked = :blocked";
        $params[':blocked'] = $blocked;
    }

    if (empty($updateFields)) {
        throw new Exception('No update parameters provided');
    }

    $query = "UPDATE tbl_clients_applications 
              SET " . implode(", ", $updateFields) . "
              WHERE Client_Id = :client_id 
              AND App_Id = :app_id";
    
    $stmt = $db1->prepare($query);
    if (!$stmt) {
        throw new Exception('Prepare failed: ' . implode(' ', $db1->errorInfo()));
    }

    // Bind the common parameters
    $params[':client_id'] = $client_id;
    $params[':app_id'] = $app_id;

    // Bind all parameters
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value, is_int($value) ? PDO::PARAM_INT : PDO::PARAM_STR);
    }
    
    $result = $stmt->execute();
    
    if ($result) {
        $rowCount = $stmt->rowCount();
        if ($rowCount > 0) {
            echo json_encode([
                'success' => true,
                'message' => 'Record updated successfully',
                'affected_rows' => $rowCount
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'No matching record found to update'
            ]);
        }
    } else {
        throw new Exception('Execute failed: ' . implode(' ', $stmt->errorInfo()));
    }

} catch (Exception $e) {
    error_log("Error in update_package_limit.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ]);
}

// No need to explicitly close PDO connections 