<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../includes/config.php';
require_once '../includes/DataBaseObjects.php';

header('Content-Type: application/json');

// Start session and check authentication
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    $db = new DataBaseObjects();
    
    $invoice_id = $_GET['id'] ?? null;
    $client_id = $_GET['client_id'] ?? null;
    $status = $_GET['status'] ?? null;
    
    if ($invoice_id) {
        // Get specific invoice with items
        $stmt = $db1->prepare("
            SELECT i.*, c.CompanyName AS client_name, c.Email AS email, c.Telephone AS phone
            FROM invoices i
            LEFT JOIN tbl_clients c ON i.client_id = c.Id
            WHERE i.id = ?
        ");
        $stmt->execute([$invoice_id]);
        $invoice = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($invoice)) {
            http_response_code(404);
            echo json_encode(['error' => 'Invoice not found']);
            exit;
        }
        
        $invoice = $invoice[0];
        
        // Get invoice items
        $stmt = $db1->prepare("
            SELECT ii.*, a.name as application_name
            FROM invoice_items ii
            LEFT JOIN applications a ON ii.application_id = a.id
            WHERE ii.invoice_id = ?
            ORDER BY ii.id
        ");
        $stmt->execute([$invoice_id]);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get payment allocations
        $stmt = $db1->prepare("
            SELECT pa.*, p.payment_date, p.method, p.reference
            FROM payment_allocations pa
            LEFT JOIN payments p ON pa.payment_id = p.id
            WHERE pa.invoice_id = ?
            ORDER BY p.payment_date
        ");
        $stmt->execute([$invoice_id]);
        $allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $invoice['items'] = $items;
        $invoice['allocations'] = $allocations;
        
        echo json_encode(['success' => true, 'invoice' => $invoice]);
        
    } else {
        // Get list of invoices
        $where_conditions = [];
        $params = [];
        
        if ($client_id) {
            $where_conditions[] = "i.client_id = ?";
            $params[] = $client_id;
        }
        
        if ($status) {
            $where_conditions[] = "i.status = ?";
            $params[] = $status;
        }
        
        $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
        
        $stmt = $db1->prepare("
            SELECT i.*, c.CompanyName AS client_name,
                   COUNT(ii.id) as item_count,
                   SUM(ii.line_subtotal) as calculated_total
            FROM invoices i
            LEFT JOIN tbl_clients c ON i.client_id = c.Id
            LEFT JOIN invoice_items ii ON i.id = ii.invoice_id
            {$where_clause}
            GROUP BY i.id
            ORDER BY i.issue_date DESC
        ");
        $stmt->execute($params);
        $invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode(['success' => true, 'invoices' => $invoices]);
    }
    
} catch (Exception $e) {
    error_log("Invoices Get API - Error: " . $e->getMessage());
    error_log("Invoices Get API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(400);
    echo json_encode([
        'error' => $e->getMessage(),
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
}
?>
