<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get clients and applications for form using existing $db1 connection
$clients = [];
$applications = [];

try {
    $stmt = $db1->prepare("SELECT id, CompanyName FROM tbl_clients ORDER BY CompanyName");
    $stmt->execute();
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Handle error silently for now
    $clients = [];
}

try {
    $stmt = $db1->prepare("SELECT id, Application_Name name FROM tbl_applications Where visible = 1 ORDER BY Application_Name");
    $stmt->execute();
    $applications = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Handle error silently for now
    $applications = [];
}

// Form submission is now handled via AJAX
?>

<!-- Create Invoice Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Create Invoice</h2>
    <a href="?page=invoices" class="btn btn-secondary">Back to Invoices</a>
</div>

<div id="errorAlert" class="alert alert-danger" style="display: none;"></div>

<form method="POST" id="invoiceForm" onsubmit="submitInvoice(event)">
    <div class="row">
        <div class="col-md-8">
            <div class="card mb-4">
                <div class="card-header">
                    <h5>Invoice Details</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Client *</label>
                            <select name="client_id" class="form-select" required>
                                <option value="">Select Client</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?= $client['id'] ?>" <?= isset($_POST['client_id']) && $_POST['client_id'] == $client['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($client['CompanyName']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Currency *</label>
                            <select name="currency" class="form-select" required>
                                <option value="USD" <?= isset($_POST['currency']) && $_POST['currency'] == 'USD' ? 'selected' : '' ?>>USD</option>
                                <option value="SAR" <?= isset($_POST['currency']) && $_POST['currency'] == 'SAR' ? 'selected' : '' ?>>SAR</option>
                                <option value="EGP" <?= isset($_POST['currency']) && $_POST['currency'] == 'EGP' ? 'selected' : '' ?>>EGP</option>
                                <option value="EUR" <?= isset($_POST['currency']) && $_POST['currency'] == 'EUR' ? 'selected' : '' ?>>EUR</option>
                                <option value="GBP" <?= isset($_POST['currency']) && $_POST['currency'] == 'GBP' ? 'selected' : '' ?>>GBP</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Issue Date *</label>
                            <input type="date" name="issue_date" class="form-control" value="<?= $_POST['issue_date'] ?? date('Y-m-d') ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Due Date</label>
                            <input type="date" name="due_date" class="form-control" value="<?= $_POST['due_date'] ?? '' ?>">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Notes</label>
                            <textarea name="notes" class="form-control" rows="2"><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Invoice Items -->
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5>Invoice Items</h5>
                    <button type="button" class="btn btn-sm btn-success" onclick="addItem()">Add Item</button>
                </div>
                <div class="card-body">
                    <div id="itemsContainer">
                        <div class="item-row row g-3 mb-3">
                            <div class="col-md-4">
                                <label class="form-label">Application *</label>
                                <select name="application_id[]" class="form-select" required>
                                    <option value="">Select Application</option>
                                    <?php foreach ($applications as $app): ?>
                                        <option value="<?= $app['id'] ?>"><?= htmlspecialchars($app['name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Description *</label>
                                <input type="text" name="description[]" class="form-control" required>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Credit Units *</label>
                                <input type="number" name="credit_units[]" class="form-control" min="1" required>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Unit Price *</label>
                                <input type="number" name="unit_price[]" class="form-control" step="0.01" min="0" required>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Quantity</label>
                                <input type="number" name="quantity[]" class="form-control" step="0.01" min="0.01" value="1">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Expiry Date</label>
                                <input type="date" name="expiry_date[]" class="form-control">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Line Total</label>
                                <input type="text" class="form-control line-total" readonly>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <button type="button" class="btn btn-sm btn-danger d-block" onclick="removeItem(this)">Remove</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-4">
            <div class="card">
                <div class="card-header">
                    <h5>Summary</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <strong>Subtotal:</strong> <span id="subtotal">0.00</span>
                    </div>
                    <div class="mb-3">
                        <strong>Total:</strong> <span id="total">0.00</span>
                    </div>
                    <button type="submit" class="btn btn-primary w-100">Create Invoice</button>
                </div>
            </div>
        </div>
    </div>
</form>

<script>
    function addItem() {
        const container = document.getElementById('itemsContainer');
        const itemRow = container.querySelector('.item-row').cloneNode(true);
        
        // Clear values
        itemRow.querySelectorAll('input, select').forEach(input => {
            if (input.type === 'date' || input.type === 'text' || input.type === 'number') {
                input.value = '';
            } else if (input.tagName === 'SELECT') {
                input.selectedIndex = 0;
            }
        });
        
        // Add event listeners
        addItemEventListeners(itemRow);
        container.appendChild(itemRow);
    }
    
    function removeItem(button) {
        const itemsContainer = document.getElementById('itemsContainer');
        if (itemsContainer.children.length > 1) {
            button.closest('.item-row').remove();
            calculateTotals();
        }
    }
    
    function addItemEventListeners(itemRow) {
        const quantityInput = itemRow.querySelector('input[name="quantity[]"]');
        const unitPriceInput = itemRow.querySelector('input[name="unit_price[]"]');
        const lineTotalInput = itemRow.querySelector('.line-total');
        
        [quantityInput, unitPriceInput].forEach(input => {
            input.addEventListener('input', () => {
                const quantity = parseFloat(quantityInput.value) || 0;
                const unitPrice = parseFloat(unitPriceInput.value) || 0;
                lineTotalInput.value = (quantity * unitPrice).toFixed(2);
                calculateTotals();
            });
        });
    }
    
    function calculateTotals() {
        let subtotal = 0;
        document.querySelectorAll('.line-total').forEach(input => {
            subtotal += parseFloat(input.value) || 0;
        });
        
        document.getElementById('subtotal').textContent = subtotal.toFixed(2);
        document.getElementById('total').textContent = subtotal.toFixed(2);
    }
    
    // Initialize event listeners for first row
    document.addEventListener('DOMContentLoaded', function() {
        addItemEventListeners(document.querySelector('.item-row'));
    });
    
    function showError(message) {
        const errorAlert = document.getElementById('errorAlert');
        errorAlert.textContent = message;
        errorAlert.style.display = 'block';
        errorAlert.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
    
    function submitInvoice(event) {
        event.preventDefault();
        
        const form = document.getElementById('invoiceForm');
        const formData = new FormData(form);
        
        // Show loading state
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.textContent = 'Creating...';
        submitBtn.disabled = true;
        
        // Convert form data to JSON
        const invoiceData = {
            client_id: formData.get('client_id'),
            currency: formData.get('currency'),
            issue_date: formData.get('issue_date'),
            due_date: formData.get('due_date') || null,
            notes: formData.get('notes') || '',
            items: []
        };
        
        // Collect items
        const itemRows = document.querySelectorAll('.item-row');
        itemRows.forEach(row => {
            const description = row.querySelector('input[name="description[]"]').value;
            const creditUnits = row.querySelector('input[name="credit_units[]"]').value;
            const unitPrice = row.querySelector('input[name="unit_price[]"]').value;
            
            if (description && creditUnits && unitPrice) {
                invoiceData.items.push({
                    application_id: row.querySelector('select[name="application_id[]"]').value,
                    description: description,
                    credit_units: parseInt(creditUnits),
                    expiry_date: row.querySelector('input[name="expiry_date[]"]').value || null,
                    unit_price: parseFloat(unitPrice),
                    quantity: parseFloat(row.querySelector('input[name="quantity[]"]').value) || 1
                });
            }
        });
        
        if (invoiceData.items.length === 0) {
            showError('At least one item is required');
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
            return;
        }
        
        // Send AJAX request
        fetch('../api/invoices_create.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(invoiceData),
            credentials: 'same-origin'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                window.location.href = '?page=invoice_view&id=' + data.invoice_id;
            } else {
                showError('Error: ' + (data.error || 'Unknown error occurred'));
                submitBtn.textContent = originalText;
                submitBtn.disabled = false;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showError('Failed to create invoice. Please try again.');
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
        });
    }
</script>
