<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get payment ID from URL
$payment_id = $_GET['id'] ?? 0;

if (!$payment_id) {
    echo '<div class="alert alert-danger">Payment ID is required</div>';
    return;
}

// Get payment details
$payment = null;
$unpaid_invoices = [];
$existing_allocations = [];

try {
    // Get payment header
    $stmt = $db1->prepare("
        SELECT p.*, c.CompanyName as client_name
        FROM payments p
        LEFT JOIN tbl_clients c ON p.client_id = c.Id
        WHERE p.id = ?
    ");
    $stmt->execute([$payment_id]);
    $payment = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$payment) {
        echo '<div class="alert alert-danger">Payment not found</div>';
        return;
    }
    
    // Get existing allocations
    $stmt = $db1->prepare("
        SELECT pa.*, i.invoice_number, i.issue_date, i.total as invoice_total
        FROM payment_allocations pa
        LEFT JOIN invoices i ON pa.invoice_id = i.id
        WHERE pa.payment_id = ?
        ORDER BY pa.created_at
    ");
    $stmt->execute([$payment_id]);
    $existing_allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate allocated amount
    $allocated_amount = array_sum(array_column($existing_allocations, 'amount'));
    $remaining_amount = $payment['amount'] - $allocated_amount;
    
    if ($remaining_amount <= 0) {
        echo '<div class="alert alert-info">This payment has been fully allocated.</div>';
        echo '<a href="?page=payment_view&id=' . $payment_id . '" class="btn btn-secondary">View Payment</a>';
        return;
    }
    
    // Get unpaid invoices for this client
    $stmt = $db1->prepare("
        SELECT i.*, 
               COALESCE(SUM(pa.amount), 0) as paid_amount,
               (i.total - COALESCE(SUM(pa.amount), 0)) as unpaid_amount
        FROM invoices i
        LEFT JOIN payment_allocations pa ON i.id = pa.invoice_id
        WHERE i.client_id = ? 
          AND i.status IN ('issued', 'partially_paid')
        GROUP BY i.id
        HAVING unpaid_amount > 0
        ORDER BY i.issue_date ASC
    ");
    $stmt->execute([$payment['client_id']]);
    $unpaid_invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    echo '<div class="alert alert-danger">Error loading payment: ' . htmlspecialchars($e->getMessage()) . '</div>';
    return;
}
?>

<!-- Payment Allocation Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Allocate Payment</h2>
    <div>
        <a href="?page=payment_view&id=<?= $payment['id'] ?>" class="btn btn-secondary">View Payment</a>
        <a href="?page=payments" class="btn btn-outline-secondary">Back to Payments</a>
    </div>
</div>

<!-- Payment Summary -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Payment Summary</h5>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <p><strong>Client:</strong> <?= htmlspecialchars($payment['client_name']) ?></p>
                <p><strong>Payment Date:</strong> <?= $payment['payment_date'] ?></p>
                <p><strong>Method:</strong> 
                    <span class="badge bg-info"><?= ucfirst($payment['method']) ?></span>
                </p>
            </div>
            <div class="col-md-6">
                <p><strong>Total Amount:</strong> <?= number_format($payment['amount'], 2) ?> <?= $payment['currency'] ?></p>
                <p><strong>Allocated:</strong> <?= number_format($allocated_amount, 2) ?> <?= $payment['currency'] ?></p>
                <p><strong>Remaining:</strong> 
                    <span class="badge bg-warning"><?= number_format($remaining_amount, 2) ?> <?= $payment['currency'] ?></span>
                </p>
            </div>
        </div>
    </div>
</div>

<!-- Existing Allocations -->
<?php if (!empty($existing_allocations)): ?>
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Existing Allocations</h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Invoice #</th>
                        <th>Issue Date</th>
                        <th>Invoice Total</th>
                        <th>Amount Allocated</th>
                        <th>Date Allocated</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($existing_allocations as $allocation): ?>
                        <tr>
                            <td>
                                <a href="?page=invoice_view&id=<?= $allocation['invoice_id'] ?>">
                                    <?= htmlspecialchars($allocation['invoice_number']) ?>
                                </a>
                            </td>
                            <td><?= $allocation['issue_date'] ?></td>
                            <td><?= number_format($allocation['invoice_total'], 2) ?></td>
                            <td><?= number_format($allocation['amount'], 2) ?></td>
                            <td><?= $allocation['created_at'] ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Allocate to Invoices -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Allocate to Invoices</h5>
    </div>
    <div class="card-body">
        <?php if (empty($unpaid_invoices)): ?>
            <div class="alert alert-info">
                No unpaid invoices found for this client.
            </div>
        <?php else: ?>
            <form id="allocationForm" method="POST" onsubmit="submitAllocation(event)">
                <input type="hidden" name="payment_id" value="<?= $payment['id'] ?>">
                <input type="hidden" name="remaining_amount" value="<?= $remaining_amount ?>">
                
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Invoice #</th>
                                <th>Issue Date</th>
                                <th>Invoice Total</th>
                                <th>Paid Amount</th>
                                <th>Unpaid Amount</th>
                                <th>Allocate Amount</th>
                                <th>Currency</th>
                                <th>Rate to <?= htmlspecialchars($payment['currency']) ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($unpaid_invoices as $invoice): ?>
                                <tr data-unpaid="<?= htmlspecialchars($invoice['unpaid_amount']) ?>">
                                    <td>
                                        <a href="?page=invoice_view&id=<?= $invoice['id'] ?>">
                                            <?= htmlspecialchars($invoice['invoice_number']) ?>
                                        </a>
                                    </td>
                                    <td><?= $invoice['issue_date'] ?></td>
                                    <td><?= number_format($invoice['total'], 2) ?></td>
                                    <td><?= number_format($invoice['paid_amount'], 2) ?></td>
                                    <td><?= number_format($invoice['unpaid_amount'], 2) ?></td>
                                    <td>
                                        <input type="number" 
                                               name="allocations[<?= $invoice['id'] ?>]" 
                                               class="form-control allocation-amount" 
                                               min="0" 
                                               step="0.01" 
                                               placeholder="0.00"
                                               oninput="updateRemainingAmount()" onchange="updateRemainingAmount()">
                                    </td>
                                    <td style="min-width:120px;">
                                        <select class="form-select form-select-sm allocation-currency" data-invoice-id="<?= $invoice['id'] ?>" onchange="onCurrencyChange(this)">
                                            <!-- options loaded by JS -->
                                        </select>
                                    </td>
                                    <td style="min-width:150px;">
                                        <input type="number" step="any" min="0" class="form-control form-control-sm allocation-rate" data-invoice-id="<?= $invoice['id'] ?>" onchange="updateRemainingAmount()" placeholder="Units per USD">
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                        <tfoot>
                            <tr class="table-info">
                                <td colspan="5"><strong>Total to Allocate (in payment currency <?= htmlspecialchars($payment['currency']) ?>):</strong></td>
                                <td>
                                    <span id="totalToAllocate">0.00</span>
                                </td>
                                <td colspan="2"></td>
                            </tr>
                            <tr class="table-<?= $remaining_amount > 0 ? 'warning' : 'success' ?>">
                                <td colspan="5"><strong>Remaining Payment:</strong></td>
                                <td>
                                    <span id="remainingPayment"><?= number_format($remaining_amount, 2) ?></span>
                                </td>
                                <td colspan="2"></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
                
                <div class="d-flex justify-content-between mt-3">
                    <button type="button" class="btn btn-secondary" onclick="clearAllocations()">Clear All</button>
                    <div class="text-end">
                        <div id="allocateHint" class="text-danger small mb-1" style="display:none;"></div>
                        <button type="submit" class="btn btn-primary" id="submitBtn" disabled>Allocate Payment</button>
                    </div>
                </div>
            </form>
        <?php endif; ?>
    </div>
</div>

<script>
let currencyRates = {};
const paymentCurrency = '<?= $payment['currency'] ?>';

function getUnitsPerUsd(code) {
    return parseFloat(currencyRates[code]) || 0;
}

function round2(n) {
    return Math.round((n + Number.EPSILON) * 100) / 100;
}

function onCurrencyChange(selectEl) {
    const tr = selectEl.closest('tr');
    const rateInput = tr.querySelector('.allocation-rate');
    const code = selectEl.value;
    const allocUnits = getUnitsPerUsd(code);
    const payUnits = getUnitsPerUsd(paymentCurrency);
    if (allocUnits > 0 && payUnits > 0) {
        // rate = payment currency per 1 allocation currency
        const rate = payUnits / allocUnits;
        rateInput.value = rate.toFixed(6);
    }
    updateRemainingAmount();
}

function updateRemainingAmount() {
    let totalAllocated = 0;
    const rows = document.querySelectorAll('tbody tr');
    const reasons = [];

    rows.forEach(tr => {
        const amount = parseFloat(tr.querySelector('.allocation-amount')?.value) || 0;
        const curSel = tr.querySelector('.allocation-currency');
        const rateInput = tr.querySelector('.allocation-rate');
        const allocCurrency = curSel ? curSel.value : paymentCurrency;
        const explicitRate = parseFloat(rateInput?.value);
        const derivedRate = (() => {
            const allocUnits = getUnitsPerUsd(allocCurrency);
            const payUnits = getUnitsPerUsd(paymentCurrency);
            if (allocUnits > 0 && payUnits > 0) return payUnits / allocUnits;
            return 0;
        })();
        const rateToPayment = !isNaN(explicitRate) && explicitRate > 0 ? explicitRate : derivedRate;
        if (amount > 0 && rateToPayment > 0) {
            const amtInPayment = round2(amount * rateToPayment);
            totalAllocated = round2(totalAllocated + amtInPayment);
        } else if (amount > 0 && rateToPayment <= 0) {
            reasons.push('Missing/invalid rate for an allocation row');
        }

        // Row-level validation against invoice unpaid in row currency
        const unpaid = parseFloat(tr.getAttribute('data-unpaid')) || 0;
        if (amount > unpaid + 1e-9) {
            reasons.push('Row exceeds invoice unpaid amount');
        }
    });

    const remainingAmount = round2(<?= $remaining_amount ?> - totalAllocated);

    document.getElementById('totalToAllocate').textContent = totalAllocated.toFixed(2);
    document.getElementById('remainingPayment').textContent = remainingAmount.toFixed(2);
    
    // Enable/disable submit button
    const submitBtn = document.getElementById('submitBtn');
    const hint = document.getElementById('allocateHint');
    hint.style.display = 'none';
    hint.textContent = '';

    if (totalAllocated > 0 && remainingAmount >= 0 && reasons.length === 0) {
        submitBtn.disabled = false;
    } else {
        submitBtn.disabled = true;
        if (totalAllocated <= 0) reasons.push('Enter allocation amounts');
        if (remainingAmount < 0) reasons.push('Total exceeds remaining payment');
        if (reasons.length) {
            hint.textContent = reasons[0];
            hint.style.display = '';
        }
    }
    
    // Update row colors
    const allocationInputs = document.querySelectorAll('.allocation-amount');
    allocationInputs.forEach(input => {
        const row = input.closest('tr');
        const allocated = parseFloat(input.value) || 0;
        if (allocated > 0) {
            row.classList.add('table-success');
        } else {
            row.classList.remove('table-success');
        }
    });
}

function clearAllocations() {
    const allocationInputs = document.querySelectorAll('.allocation-amount');
    allocationInputs.forEach(input => {
        input.value = '';
    });
    updateRemainingAmount();
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', async function() {
    // Load rates
    try {
        const res = await fetch('api/currency_rates.php', { credentials: 'same-origin' });
        const data = await res.json();
        const map = {};
        (data.rates || []).forEach(r => { map[r.currency_code] = parseFloat(r.units_per_usd); });
        currencyRates = map;
    } catch (e) { currencyRates = {}; }

    // Populate currency selects
    const currencies = Object.keys(currencyRates).length ? Object.keys(currencyRates) : ['USD','SAR','EGP','EUR','GBP'];
    document.querySelectorAll('.allocation-currency').forEach(sel => {
        sel.innerHTML = '';
        currencies.forEach(code => {
            const opt = document.createElement('option');
            opt.value = code;
            opt.textContent = code;
            sel.appendChild(opt);
        });
        // default to payment currency
        sel.value = paymentCurrency;
        onCurrencyChange(sel);
    });

    updateRemainingAmount();
});

// Validation handled in submitAllocation()

function submitAllocation(event) {
    event.preventDefault();
    
    const form = document.getElementById('allocationForm');
    const formData = new FormData(form);
    
    // Show loading state
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Allocating...';
    submitBtn.disabled = true;
    
    // Collect allocation data
    const allocationData = {
        payment_id: formData.get('payment_id'),
        remaining_amount: parseFloat(formData.get('remaining_amount')),
        allocations: []
    };
    
    // Collect allocations
    const rows = document.querySelectorAll('tbody tr');
    let totalAllocated = 0;
    rows.forEach(tr => {
        const amount = parseFloat(tr.querySelector('.allocation-amount')?.value) || 0;
        if (amount <= 0) { return; }
        const nameAttr = tr.querySelector('.allocation-amount').getAttribute('name');
        const invoiceId = nameAttr.match(/\[(\d+)\]/)[1];
        const currency = tr.querySelector('.allocation-currency')?.value || paymentCurrency;
        const rateInput = tr.querySelector('.allocation-rate');
        const explicitRate = parseFloat(rateInput?.value);
        const derivedRate = (() => {
            const allocUnits = getUnitsPerUsd(currency);
            const payUnits = getUnitsPerUsd(paymentCurrency);
            if (allocUnits > 0 && payUnits > 0) return payUnits / allocUnits;
            return 0;
        })();
        const rateToPayment = !isNaN(explicitRate) && explicitRate > 0 ? explicitRate : derivedRate;
        const unitsPerUsd = getUnitsPerUsd(currency);

        allocationData.allocations.push({ invoice_id: parseInt(invoiceId,10), amount: amount, currency: currency, units_per_usd: unitsPerUsd, rate_to_payment: rateToPayment });
        // sum in payment currency for validation
        if (rateToPayment > 0) {
            totalAllocated = round2(totalAllocated + round2(amount * rateToPayment));
        }
    });
    
    // Validate
    if (totalAllocated <= 0) {
        alert('Please allocate at least some amount to invoices.');
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        return;
    }
    
    if (round2(totalAllocated) > round2(allocationData.remaining_amount)) {
        alert('You cannot allocate more than the remaining payment amount.');
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        return;
    }
    
    if (!confirm('Are you sure you want to allocate this payment? This action cannot be undone.')) {
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        return;
    }
    
    // Send AJAX request
    fetch('../api/payments_allocate.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(allocationData),
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Payment allocated successfully!');
            window.location.href = '?page=payment_view&id=' + allocationData.payment_id;
        } else {
            alert('Error: ' + (data.error || 'Unknown error occurred'));
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Failed to allocate payment. Please try again.');
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
    });
}
</script>
