<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get clients for form using existing $db1 connection
$clients = [];
$accounts = [];

try {
    $stmt = $db1->prepare("SELECT id, CompanyName FROM tbl_clients ORDER BY CompanyName");
    $stmt->execute();
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Handle error silently for now
    $clients = [];
}

// Fetch accounts for selection (bank, wallet, cashbox)
try {
    $stmtAcc = $db1->prepare("SELECT id, name, type FROM accounts WHERE is_active = 1 ORDER BY name");
    $stmtAcc->execute();
    $accounts = $stmtAcc->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $accounts = [];
}

        // Form submission is now handled via AJAX
?>

<!-- Record Payment Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Record Payment</h2>
    <a href="?page=payments" class="btn btn-secondary">Back to Payments</a>
</div>

<div id="errorAlert" class="alert alert-danger" style="display: none;"></div>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5>Payment Details</h5>
            </div>
            <div class="card-body">
                <form method="POST" id="paymentForm" onsubmit="submitPayment(event)">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Client *</label>
                            <input type="text" id="clientSearch" class="form-control mb-2" placeholder="Search clients...">
                            <select name="client_id" class="form-select" required>
                                <option value="">Select Client</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?= $client['id'] ?>" <?= isset($_POST['client_id']) && $_POST['client_id'] == $client['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($client['CompanyName']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Payment Date *</label>
                            <input type="date" name="payment_date" class="form-control" value="<?= $_POST['payment_date'] ?? date('Y-m-d') ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Amount *</label>
                            <input type="number" name="amount" class="form-control" step="0.01" min="0.01" value="<?= $_POST['amount'] ?? '' ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Currency *</label>
                            <select name="currency" class="form-select" required>
                                <option value="USD" <?= isset($_POST['currency']) && $_POST['currency'] == 'USD' ? 'selected' : '' ?>>USD</option>
                                <option value="SAR" <?= isset($_POST['currency']) && $_POST['currency'] == 'SAR' ? 'selected' : '' ?>>SAR</option>
                                <option value="EGP" <?= isset($_POST['currency']) && $_POST['currency'] == 'EGP' ? 'selected' : '' ?>>EGP</option>
                                <option value="EUR" <?= isset($_POST['currency']) && $_POST['currency'] == 'EUR' ? 'selected' : '' ?>>EUR</option>
                                <option value="GBP" <?= isset($_POST['currency']) && $_POST['currency'] == 'GBP' ? 'selected' : '' ?>>GBP</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Payment Method *</label>
                            <select name="method" class="form-select" required>
                                <option value="">Select Method</option>
                                <option value="cash" <?= isset($_POST['method']) && $_POST['method'] == 'cash' ? 'selected' : '' ?>>Cash</option>
                                <option value="bank" <?= isset($_POST['method']) && $_POST['method'] == 'bank' ? 'selected' : '' ?>>Bank Transfer</option>
                                <option value="card" <?= isset($_POST['method']) && $_POST['method'] == 'card' ? 'selected' : '' ?>>Credit/Debit Card</option>
                                <option value="transfer" <?= isset($_POST['method']) && $_POST['method'] == 'transfer' ? 'selected' : '' ?>>Wire Transfer</option>
                                <option value="wallet" <?= isset($_POST['method']) && $_POST['method'] == 'wallet' ? 'selected' : '' ?>>Wallet</option>
                                <option value="other" <?= isset($_POST['method']) && $_POST['method'] == 'other' ? 'selected' : '' ?>>Other</option>
                            </select>
                        </div>
                        <div class="col-md-8">
                            <label class="form-label">Account *</label>
                            <select name="account_id" class="form-select" required>
                                <option value="">Select Account</option>
                                <?php foreach ($accounts as $acc): ?>
                                    <option value="<?= $acc['id'] ?>" data-type="<?= htmlspecialchars($acc['type']) ?>" <?= isset($_POST['account_id']) && $_POST['account_id'] == $acc['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($acc['name']) ?> (<?= htmlspecialchars(strtoupper($acc['type'])) ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Reference</label>
                            <input type="text" name="reference" class="form-control" value="<?= htmlspecialchars($_POST['reference'] ?? '') ?>" placeholder="Transaction ID, Check #, etc.">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Notes</label>
                            <textarea name="notes" class="form-control" rows="2" placeholder="Additional payment details"><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                        </div>
                        <div class="col-12">
                            <button type="submit" class="btn btn-primary">Record Payment</button>
                            <a href="?page=payments" class="btn btn-outline-secondary ms-2">Cancel</a>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h5>Next Steps</h5>
            </div>
            <div class="card-body">
                <p>After recording the payment, you'll be redirected to allocate it to specific invoices.</p>
                <ul class="list-unstyled">
                    <li><i class="bi bi-check-circle text-success"></i> Payment recorded</li>
                    <li><i class="bi bi-arrow-right text-primary"></i> Allocate to invoices</li>
                    <li><i class="bi bi-check-circle text-success"></i> Update invoice status</li>
                </ul>
            </div>
        </div>
        
        <?php if (DEVELOPMENT_MODE): ?>
        <div class="card mt-3">
            <div class="card-header">
                <h5>API Status</h5>
            </div>
            <div class="card-body">
                <button type="button" class="btn btn-sm btn-outline-info" onclick="testAPI()">Test API Connection</button>
                <div id="api-status" class="mt-2"></div>
            </div>
        </div>
        
        <div class="card mt-3">
            <div class="card-header">
                <h5>Recent Error Logs</h5>
            </div>
            <div class="card-body">
                <button type="button" class="btn btn-sm btn-outline-warning" onclick="viewLogs()">View Recent Logs</button>
                <div id="log-content" class="mt-2"></div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php if (DEVELOPMENT_MODE): ?>
<script>
function testAPI() {
    const statusDiv = document.getElementById('api-status');
    statusDiv.innerHTML = '<div class="text-muted">Testing API...</div>';
    
    fetch('../api/payments_create.php', {
        method: 'GET',
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            statusDiv.innerHTML = `
                <div class="alert alert-success">
                    <strong>✓ API Working</strong><br>
                    Session: ${data.session_status}<br>
                    Admin ID: ${data.admin_id || 'Not set'}
                </div>
            `;
        } else {
            statusDiv.innerHTML = `
                <div class="alert alert-danger">
                    <strong>✗ API Error</strong><br>
                    ${data.error || 'Unknown error'}
                </div>
            `;
        }
    })
    .catch(error => {
        statusDiv.innerHTML = `
            <div class="alert alert-danger">
                <strong>✗ Connection Failed</strong><br>
            ${error.message}
            </div>
        `;
    });
}

function viewLogs() {
    const logDiv = document.getElementById('log-content');
    logDiv.innerHTML = '<div class="text-muted">Loading logs...</div>';
    
    // For now, show a message about where to find logs
    logDiv.innerHTML = `
        <div class="alert alert-info">
            <strong>Error Logs Location:</strong><br>
            Check your server's error log files:<br>
            • Apache: /var/log/apache2/error.log<br>
            • Nginx: /var/log/nginx/error.log<br>
            • PHP: /var/log/php_errors.log<br>
            • Custom: Check your php.ini error_log setting<br><br>
            <strong>Recent API calls are logged with:</strong><br>
            • "Payment creation request:"<br>
            • "Payment API Response:"<br>
            • "Payment API Error:"<br>
            • "Payments API -" (various messages)
        </div>
    `;
}
</script>
<?php endif; ?>

<script>
function showError(message) {
    const errorAlert = document.getElementById('errorAlert');
    errorAlert.textContent = message;
    errorAlert.style.display = 'block';
    errorAlert.scrollIntoView({ behavior: 'smooth', block: 'center' });
}

function submitPayment(event) {
    event.preventDefault();
    
    const form = document.getElementById('paymentForm');
    const formData = new FormData(form);
    
    // Show loading state
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Recording...';
    submitBtn.disabled = true;
    
    // Convert form data to JSON
    const paymentData = {
        client_id: formData.get('client_id'),
        amount: parseFloat(formData.get('amount')),
        currency: formData.get('currency'),
        payment_date: formData.get('payment_date'),
        method: formData.get('method'),
        account_id: formData.get('account_id'),
        reference: formData.get('reference') || '',
        notes: formData.get('notes') || ''
    };
    
    // Validate required fields
    if (!paymentData.client_id || paymentData.amount <= 0 || !paymentData.currency || !paymentData.method || !paymentData.account_id) {
        showError('Please fill in all required fields');
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        return;
    }
    
    // Send AJAX request
    fetch('../api/payments_create.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(paymentData),
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            window.location.href = '?page=payment_allocate&id=' + data.payment_id;
        } else {
            showError('Error: ' + (data.error || 'Unknown error occurred'));
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showError('Failed to record payment. Please try again.');
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
    });
}

// Make client select searchable using the input above it
document.addEventListener('DOMContentLoaded', function() {
	var searchInput = document.getElementById('clientSearch');
	var selectEl = document.querySelector('select[name="client_id"]');
	var methodSelect = document.querySelector('select[name="method"]');
	var accountSelect = document.querySelector('select[name="account_id"]');
	if (!searchInput || !selectEl) return;

	function filterOptions() {
		var query = (searchInput.value || '').toLowerCase();
		var firstVisibleIndex = -1;
		var visibleCount = 0;

		for (var i = 0; i < selectEl.options.length; i++) {
			var opt = selectEl.options[i];
			if (opt.value === '') { // keep placeholder visible
				opt.style.display = '';
				continue;
			}
			var text = (opt.textContent || opt.innerText || '').toLowerCase();
			var isMatch = text.indexOf(query) !== -1;
			opt.style.display = isMatch ? '' : 'none';
			if (isMatch) {
				visibleCount++;
				if (firstVisibleIndex === -1) firstVisibleIndex = i;
			}
		}

		// If only one match remains, auto-select it
		if (visibleCount === 1 && firstVisibleIndex !== -1) {
			selectEl.selectedIndex = firstVisibleIndex;
		} else {
			// If current selection is hidden by filter, clear to placeholder
			var current = selectEl.options[selectEl.selectedIndex];
			if (current && current.value !== '' && current.style.display === 'none') {
				selectEl.selectedIndex = 0;
			}
		}

		return firstVisibleIndex;
	}

	searchInput.addEventListener('input', function() {
		filterOptions();
	});

	// Press Enter to select the first visible match and focus the select
	searchInput.addEventListener('keydown', function(e) {
		if (e.key === 'Enter') {
			e.preventDefault();
			var firstIndex = filterOptions();
			if (firstIndex !== -1) {
				selectEl.selectedIndex = firstIndex;
			}
			selectEl.focus();
		}
		if (e.key === 'ArrowDown') {
			// Move focus into the select to continue navigating
			selectEl.focus();
		}
	});

	// Filter accounts based on selected method
	function filterAccountsByMethod() {
		if (!methodSelect || !accountSelect) return;
		var method = methodSelect.value || '';
		var allowedTypes = null; // null means show all
		if (method === 'cash') {
			allowedTypes = ['cashbox'];
		} else if (method === 'wallet') {
			allowedTypes = ['wallet'];
		} else if (method === 'bank' || method === 'transfer') {
			allowedTypes = ['bank'];
		}

		var firstVisibleIndex = -1;
		for (var i = 0; i < accountSelect.options.length; i++) {
			var opt = accountSelect.options[i];
			if (opt.value === '') {
				opt.style.display = '';
				continue;
			}
			var type = (opt.getAttribute('data-type') || '').toLowerCase();
			var show = !allowedTypes || allowedTypes.indexOf(type) !== -1;
			opt.style.display = show ? '' : 'none';
			if (show && firstVisibleIndex === -1) firstVisibleIndex = i;
		}

		// Reset selection if current is hidden
		var current = accountSelect.options[accountSelect.selectedIndex];
		if (current && current.value !== '' && current.style.display === 'none') {
			accountSelect.selectedIndex = firstVisibleIndex !== -1 ? firstVisibleIndex : 0;
		}
	}

	if (methodSelect) {
		methodSelect.addEventListener('change', filterAccountsByMethod);
		// Apply once on load in case a method is preselected
		filterAccountsByMethod();
	}
});
</script>
